//******************************************************************************
//Name          : IDFT.cs
//Description   : Implementation to experience the beauty and the pitfalls of IDFT
//Author        : Dr. Oliver Kluge, 90411 Nuernberg, Germany
//Copyright(C)  : Oliver Kluge 2014 
//------------------------------------------------------------------------------
//******************************************************************************

using System;
using System.Windows.Forms;
using System.Windows.Forms.DataVisualization.Charting;
using Display;


class Spectrum
{
    const int samplerate = 128;                         // Abtastwerte je Periode
    const double PI      =   3.141592653589793;

    public double[] signal = new double[samplerate];    // Signalverlauf im Zeitbereich
    public double[] arg = new double[samplerate];
    public double[] AS = new double[samplerate];        // Amplitudenspektrum von signal(k)
    public double[] PS = new double[samplerate];        // Phasenspektrum

    public void CreateSignal(int select)
    {
        if (select >= 1 && select <= 8)
        {
            int width = 0;

            // 1/1 Puls
            if (select == 1) width = samplerate;
            // 1/2 Puls
            if (select == 2) width = samplerate / 2;
            // 1/4 Puls
            if (select == 3) width = samplerate / 4;
            // 1/8 Puls
            if (select == 4) width = samplerate / 8;
            // 1/16 Puls
            if (select == 5) width = samplerate / 16;
            // 1/32 Puls
            if (select == 6) width = samplerate / 32;
            // 1/64 Puls
            if (select == 7) width = samplerate / 64;
            // 1/128 Puls
            if (select == 8) width = samplerate / 128;

            for (int k = 0; k < samplerate; k++)
            {
                if (k < width)
                {
                    signal[k] = 1;
                }
                else
                {
                    signal[k] = 0;
                }
            }
        }

        // Deltakamm
        if (select == 9)
        {
            for (int k = 0; k < samplerate; k++)
            {
                if (k == 0 || k == 16 || k == 32 || k == 48 ||
                    k == 64 || k == 80 || k == 96 || k == 112 || k == 128)
                {
                    signal[k] = 1;
                }
                else
                {
                    signal[k] = 0;
                }
            }
        }

        // Spaltfunktion
        if (select == 10)
        {
            double f = 16.0;

            for (int k = 0; k < samplerate; k++)
            {
                signal[k] = 15 * Math.Sin(0.001 + PI * f * (k - samplerate / 2) / samplerate) / (0.001 + PI * f * (k - samplerate / 2) / samplerate);
            }
        }

        // FreqTransient
        if (select == 11)
        {
            for (int k = 0; k < samplerate; k++)
            {
                if (k < samplerate / 2)
                {
                    signal[k] = Math.Sin(2 * PI * 52 * k / samplerate);
                }
                if (k >= 3 * samplerate / 4 && k < (3 * samplerate / 4 + samplerate / 16))
                {
                    signal[k] = Math.Sin(2 * PI * 23 * k / samplerate);
                }
            }
        }

        // Sinus mit T <> ganzzahlig
        if (select == 12)
        {
            for (int k = 0; k < samplerate; k++)
            {
                signal[k] = Math.Sin(2 * PI * 12 * 0.95 * k / samplerate);
            }
        }

        // Sinus mit T = ganzzahlig
        if (select == 13)
        {
            for (int k = 0; k < samplerate; k++)
            {
                signal[k] = Math.Sin(2 * PI * 12 * k / samplerate);
            }
        }

        // Chirp-Signal
        if (select == 14)
        {
            double f = 0;

            for (int k = 0; k < samplerate; k++)
            {
                signal[k] = Math.Sin(2 * PI * f * k / samplerate);
                f += 1.0 / 8.0;
            }
        }

        // Walsh-Funktion 1. Ordnung
        if (select == 15)
        {
            for (int k = 0; k < samplerate; k++)
            {
                if (k < samplerate / 2)
                {
                    signal[k] = 1;
                }
                else
                {
                    signal[k] = -1;
                }
            }
        }

        // Hanning-Fenster
        if (select == 16)
        {
            for (int k = 0; k < samplerate; k++)
            {
                signal[k] = 0.5 - 0.5 * Math.Cos(2 * PI * k / samplerate);
            }
        }

        // Schalttransiente
        if (select == 17)
        {
            int tau = 0;

            for (int k = 0; k < samplerate; k++)
            {
                signal[k] = Math.Sin(2 * PI * k / samplerate);

                if (k >= 1 * samplerate / 4)
                {
                    signal[k] += 0.9 * Math.Exp(-tau / 30.0) * Math.Sin((1 + PI / 2.0) + 2 * PI * 57 * k / samplerate);
                }

                tau += 1;
            }
        }

        // store argument data
        for (int k = 0; k < samplerate; k++)
        {
            arg[k] = k;// 2 * PI * k / samplerate;
        }
    }

    public void DeleteSignal(ref double[] x)
    {
        for (int k = 0; k < x.Length; k++)
        {
            x[k] = 0;
        }
    }

    // calculate the real(!) DFT of the sampled time domain signal
    // (even though "real" and "imag" implies to deal with the complex version)
    public void DFT(double[] x)
    {
        double real = 0;
        double imag = 0;

        for (int freq = 0; freq < samplerate; freq++)
        {
            real = 0;
            imag = 0;

            for (int k = 0; k < samplerate; k++)
            {
                real += x[k] * Math.Cos(2*PI*freq*k/samplerate); // an
                imag += x[k] * Math.Sin(2*PI*freq*k/samplerate); // bn
            }
            real *= 2.0 / samplerate;
            imag *= 2.0 / samplerate;

            AS[freq] = Math.Sqrt(real * real + imag * imag);
            PS[freq] = Math.Atan2(imag, real + 1E-10);
        }
        AS[0] /= 2.0;   // Korrektur Gleichanteil
    }

    public void IDFT()
    {
        double real = 0;
        double imag = 0;

        for (int k = 0; k < samplerate; k++)
        {
            real = 0;
            imag = 0;

            for (int f = 0; f < samplerate; f++)
            {
                real += (AS[f] * Math.Cos(PS[f])) * Math.Cos(2*PI*k*f/samplerate);
                imag += (AS[f] * Math.Sin(PS[f])) * Math.Sin(2*PI*k*f/samplerate);
            }
            real /= 2.0;
            imag /= 2.0;

            signal[k] = real + imag + AS[0]/2.0;
        }
    }
}


class MainApp
{
    [STAThread]
    static void Main()
    {
        string userinput = "";
        int select = 0;

        while (select <= 17)
        {
            Spectrum mySpectrumObj = new Spectrum();

            Console.Write("Bitte Signal auswhlen:\n\n");
            Console.Write("1/1   Puls    ->  1\n");
            Console.Write("1/2   Puls    ->  2\n");
            Console.Write("1/4   Puls    ->  3\n");
            Console.Write("1/8   Puls    ->  4\n");
            Console.Write("1/16  Puls    ->  5\n");
            Console.Write("1/32  Puls    ->  6\n");
            Console.Write("1/64  Puls    ->  7\n");
            Console.Write("1/128 Puls    ->  8\n");
            Console.Write("1/8 Deltakamm ->  9\n");
            Console.Write("Spaltfunktion -> 10\n");
            Console.Write("FreqTransient -> 11\n");
            Console.Write("Sinus(T<>n)   -> 12\n");
            Console.Write("Sinus(T=n)    -> 13\n");
            Console.Write("Chirp         -> 14\n");
            Console.Write("Walsh 1.Ord.  -> 15\n");
            Console.Write("Hanning       -> 16\n");
            Console.Write("Transiente_2  -> 17\n");
            Console.WriteLine();
            userinput = Console.ReadLine();
            select = Convert.ToInt32(userinput);

            // first of all we create some time domain signal and compute the DFT
            mySpectrumObj.CreateSignal(select);
            mySpectrumObj.DFT(mySpectrumObj.signal);

            // then we delete the time domain signal, but we keep its AS[] and PS[]
            mySpectrumObj.DeleteSignal(ref mySpectrumObj.signal);

            // last but not least, we re-create the time domain signal from computing the IDFT
            mySpectrumObj.IDFT();

            // create a chart for displaying that data...
            DisplayChart mydisplay1 = new DisplayChart(720, 480, SeriesChartType.Column, "Frequency domain", "f", "|X(f)|");
            mydisplay1.AddData(mySpectrumObj.arg, mySpectrumObj.AS);

            DisplayChart mydisplay2 = new DisplayChart(720, 480, SeriesChartType.Line, "Frequency domain", "f", "Phi(f)");
            mydisplay2.AddData(mySpectrumObj.arg, mySpectrumObj.PS);

            DisplayChart mydisplay3 = new DisplayChart(720, 480, SeriesChartType.Line, "Time domain", "k", "x(k)");
            mydisplay3.AddData(mySpectrumObj.arg, mySpectrumObj.signal);

            // ...and run the chart-application
            Application.EnableVisualStyles();
            Application.Run(mydisplay1);
            Application.Run(mydisplay2);
            Application.Run(mydisplay3);

            mydisplay1.Dispose();
            mydisplay2.Dispose();
            mydisplay3.Dispose();
            Application.Exit();

            for (int i = 0; i < 20; i++) Console.WriteLine();
        }
    }
}
